/**
 * \file: mspin_lm_adapter.c
 *
 * \version: $Id:$
 *
 * \release: $Name:$
 *
 * MySpin LayerManager Adapter
 *
 * \component: MSPIN
 *
 * \author: Torsten Plate ICT-ADITG/SW2 tplate@de.adit-jv.com
 *
 * \copyright: (c) 2003 - 2013 ADIT Corporation
 *
 * \history
 * 0.1 TPlate Initial version
 *
 ***********************************************************************/

#include "mspin_lm_adapter_internal.h"
#include "mspin_lm_touch_adapter.h"
#ifdef USE_GENERATED_MYSPIN_VERSION_FILE
#include "mspin_lm_version.h"
#endif
#include "mspin_logging.h"

#include "compositor-shim.h"

#include <ilm/ilm_client.h>

#include <poll.h>
#include <pthread.h>
#include <string.h>
#include <semaphore.h>
#include <errno.h>
#include <stdlib.h>
#include <stdio.h>
#include <math.h>

#ifdef MSPIN_TOUCH_DELAY
#include "mspin_lm_delaytouch.h"
#endif

/* Note: Access to pLMContext->pWLCtx must be protected since the WL context might be deleted in parallel.
 * In general the public methods take the lock so that the internal (static) methods must not
 * take the lock a second time!
 *
 * Problem: While mspin_lm_drawFrame() is executed it is necessary that also mspin_lm_pumpEvents() is
 * executed - i.e. one should _not_ block the other.
 * Solution: Using pthread_rwlock_rdlock allows both functions to enter the critical section simultaneously.
 * All other functions use pthread_rwlock_wrlock, which assures unique access.
 *
 */

/* PRQA: Lint Message 160, 505, 578: deactivation because DLT macros generate these lint findings*/
/* PRQA: Lint Message 413, 613: Null pointer used intentionally*/
/*lint -e160 -e505 -e578 -e413 -e613*/

/* TBD crash on reconnect (SWGIII-2640) */
/* TBD eglInitialize does not return (SWGIII-2639) */
/* TBD wrong colors are displayed in RGB888 mode (SWGIII-2565) */


//Definition of callback structures for touch and pointer event listeners
static const struct wl_touch_listener mspin_lm_TouchListener = {
    mspin_lm_touchHandleDown,
    mspin_lm_touchHandleUp,
    mspin_lm_touchHandleMotion,
    mspin_lm_touchHandleFrame,
    mspin_lm_touchHandleCancel,
    NULL,
    NULL
};

static const struct wl_pointer_listener mspin_lm_PointerListener = {
    mspin_lm_pointerHandleEnter,
    mspin_lm_pointerHandleLeave,
    mspin_lm_pointerHandleMotion,
    mspin_lm_pointerHandleButton,
    mspin_lm_pointerHandleAxis,
    mspin_lm_pointerHandleFrame,
    mspin_lm_pointerHandleAxisSource,
    mspin_lm_pointerHandleAxisStop,
    mspin_lm_pointerHandleAxisDiscrete
};

static void mspin_lm_handleSeatCapabilities(void* data, struct wl_seat* pSeat, uint32_t caps)
{
    MSPIN_UNUSED(pSeat);

    WlSeat * pWlSeat = (WlSeat *)data;
    if (!data)
    {
        mspin_log_printLn(eMspinVerbosityError, "%s(data=%p) ERROR: data context is NULL",
                __FUNCTION__, data);
        return;
    }

    mspin_wl_context_t * pWLCtx = (mspin_wl_context_t *)pWlSeat->p_wl_context;
    struct wl_seat* wlSeat = pWlSeat->p_wl_seat;

    if (!wlSeat)
    {
        mspin_log_printLn(eMspinVerbosityError, "%s(wlSeat=%p) ERROR: wlSeat context is NULL",
                __FUNCTION__, wlSeat);
        return;
    }

    if (pWLCtx->flags & MSPIN_WL_ENABLE_TOUCHEVENTS)
    {
        if ((caps & WL_SEAT_CAPABILITY_TOUCH) && !pWlSeat->p_wl_touch)
        {
            mspin_log_printLn(eMspinVerbosityDebug, "%s(data=%p, seat=%p, caps=%d) -> activating touch",
                    __FUNCTION__, data, pSeat, caps);

            pWlSeat->p_wl_touch = wl_seat_get_touch(wlSeat);
            wl_touch_set_user_data(pWlSeat->p_wl_touch, pWlSeat);
            wl_touch_add_listener(pWlSeat->p_wl_touch, &mspin_lm_TouchListener, pWlSeat);
        }
        else if (!(caps & WL_SEAT_CAPABILITY_TOUCH) && pWlSeat->p_wl_touch)
        {
            mspin_log_printLn(eMspinVerbosityDebug, "%s(data=%p, seat=%p, caps=%d) -> no touch",
                    __FUNCTION__, data, pSeat, caps);
            wl_touch_destroy(pWlSeat->p_wl_touch);
            pWlSeat->p_wl_touch = NULL;
        }

        if ((caps & WL_SEAT_CAPABILITY_POINTER) && !pWlSeat->p_wl_pointer)
        {
            mspin_log_printLn(eMspinVerbosityWarn, "%s(data=%p, seat=%p, caps=%d) -> activating pointer",
                    __FUNCTION__, data, pSeat, caps);
            pWlSeat->p_wl_pointer = wl_seat_get_pointer(wlSeat);
            wl_pointer_set_user_data(pWlSeat->p_wl_pointer, pWlSeat);
            wl_pointer_add_listener(pWlSeat->p_wl_pointer, &mspin_lm_PointerListener, pWlSeat);
        }
        else if (!(caps & WL_SEAT_CAPABILITY_POINTER) && pWlSeat->p_wl_pointer)
        {
            mspin_log_printLn(eMspinVerbosityDebug, "%s(data=%p, seat=%p, caps=%d) -> no pointer",
                    __FUNCTION__, data, pSeat, caps);
            wl_pointer_destroy(pWlSeat->p_wl_pointer);
            pWlSeat->p_wl_pointer = NULL;
        }

    }
}
static void mspin_lm_handleSeatHandleName(void* data, struct wl_seat* seat, const char* name)
{
    MSPIN_UNUSED(seat);
    WlSeat * pWlSeat = (WlSeat *)data;
    free(pWlSeat->p_seat_name);
    pWlSeat->p_seat_name = strdup(name);
    mspin_log_printLn(eMspinVerbosityDebug, "%s(pWlSeat->p_seat_name has been set to: %s)",
            __FUNCTION__, pWlSeat->p_seat_name);
}
static struct wl_seat_listener mspin_lm_seatListener = {
    mspin_lm_handleSeatCapabilities,
#ifndef MSPIN_X86SIM
    mspin_lm_handleSeatHandleName
#endif
};

static const struct wl_interface *mspin_lm_wlInterfaceTypes[] = { NULL, };

static const struct wl_message mspin_lm_serverinfoRequests[] = {
        { "get_connection_id", "", mspin_lm_wlInterfaceTypes + 0 }, };

static const struct wl_message mspin_lm_serverinfoEvents[] = {
        { "connection_id", "u", mspin_lm_wlInterfaceTypes + 0 }, };

const struct wl_interface mspin_lm_serverinfoInterface = { "mspinServerInfo",
        1, ARRAY_LENGTH(mspin_lm_serverinfoRequests), mspin_lm_serverinfoRequests,
        ARRAY_LENGTH(mspin_lm_serverinfoEvents), mspin_lm_serverinfoEvents, };
/* PRQA: Lint Message 429: Pointer pWlSeat is freed in different place */
/*lint -e429*/
static void mspin_lm_addSeat(mspin_wl_context_t *p_wl_ctx, uint32_t name)
{
    struct WlSeat *pWlSeat;

    pWlSeat = malloc(sizeof *pWlSeat);
    if(pWlSeat == NULL)
    {
        mspin_log_printLn(eMspinVerbosityError, "%s() ERROR: mspin_lm_addSeat failed",
                __FUNCTION__);
        return;
    }
    pWlSeat->p_wl_touch = NULL;
    pWlSeat->p_wl_pointer = NULL;
    pWlSeat->p_seat_name = NULL;
    pWlSeat->seat_id = name;
    pWlSeat->p_wl_context = p_wl_ctx;
    pWlSeat->p_wl_seat = wl_registry_bind(p_wl_ctx->p_wl_registry, name,
                                      &wl_seat_interface, 5);
    wl_seat_add_listener(pWlSeat->p_wl_seat, &mspin_lm_seatListener, pWlSeat);
    wl_proxy_set_queue((struct wl_proxy*) pWlSeat->p_wl_seat, p_wl_ctx->p_wl_input_queue);
    pthread_mutex_lock(&p_wl_ctx->lock_seat);
    wl_list_insert(&p_wl_ctx->seats, &pWlSeat->link);
    pthread_mutex_unlock(&p_wl_ctx->lock_seat);
}
/*lint +e429*/
static void mspin_lm_register(void* data, struct wl_registry* registry,
        uint32_t name, const char* interface, uint32_t version)
{
    mspin_log_printLn(eMspinVerbosityDebug,
            "%s(data=%p, registry=%p, name=%d, interface='%s', version=%d) entered",
            __FUNCTION__, data, registry, name, interface ? interface : "n/a", version);

    mspin_layerManager_context_t *pLMContext = (mspin_layerManager_context_t*)data;
    if (!pLMContext)
    {
        mspin_log_printLn(eMspinVerbosityError, "%s(ctx=%p) ERROR: LayerManager context is NULL",
                __FUNCTION__, pLMContext);
        return;
    }

    mspin_wl_context_t *pWLCtx = pLMContext->pWLCtx;
    pWLCtx->p_lm_context = pLMContext;

    if (!pWLCtx)
    {
        mspin_log_printLn(eMspinVerbosityError, "%s(ctx=%p) ERROR: Wayland context is NULL",
                __FUNCTION__, pLMContext);
        return;
    }

    mspin_framebuffer_initGlobalHandler((void*)(pLMContext->pWLCtx), registry, name, interface, version);  //needs only Wayland context

    if (interface && (!strcmp(interface, "wl_compositor")))
    {
        pWLCtx->p_wl_compositor = (struct wl_compositor*) (wl_registry_bind(
                registry, name, &wl_compositor_interface, 1));
    }

    if (interface && (!strcmp(interface, "wl_seat")))
    {
        mspin_lm_addSeat(pWLCtx, name);
    }
}

static void mspin_lm_removeSeat(struct WlSeat *pWlSeat)
{
    if (pWlSeat)
    {
        if (pWlSeat->p_seat_name)
        {
            free(pWlSeat->p_seat_name);
            pWlSeat->p_seat_name = NULL;
        }
        if (pWlSeat->p_wl_touch)
        {
            wl_touch_destroy(pWlSeat->p_wl_touch);
            pWlSeat->p_wl_touch = NULL;
        }
        if (pWlSeat->p_wl_pointer)
        {
            wl_pointer_destroy(pWlSeat->p_wl_pointer);
            pWlSeat->p_wl_pointer = NULL;
        }
        if (pWlSeat->p_wl_seat)
        {
            wl_seat_destroy(pWlSeat->p_wl_seat);
            pWlSeat->p_wl_seat = NULL;
        }
        wl_list_remove(&pWlSeat->link);
        free(pWlSeat);
        pWlSeat = NULL;
    }
}
/* PRQA: Lint Message 40: "link" used intentionally*/
/*lint -e40*/
static void mspin_lm_remove(void *data, struct wl_registry *wl_registry, uint32_t name)
{
    mspin_log_printLn(eMspinVerbosityDebug, "%s(ctx=%p, registry=%p, name=%d)",
            __FUNCTION__, data, wl_registry, name);

    mspin_layerManager_context_t *pLMContext = (mspin_layerManager_context_t*)data;
    if (!pLMContext)
    {
        mspin_log_printLn(eMspinVerbosityError, "%s(ctx=%p) ERROR: LayerManager context is NULL",
                __FUNCTION__, pLMContext);
        return;
    }

    mspin_wl_context_t *pWLCtx = pLMContext->pWLCtx;
    if (!pWLCtx)
    {
        mspin_log_printLn(eMspinVerbosityError, "%s(ctx=%p) ERROR: Wayland context is NULL",
                __FUNCTION__, pLMContext);
        return;
    }

    MSPIN_UNUSED(wl_registry);

    pthread_mutex_lock(&pWLCtx->lock_seat);
    WlSeat *pWlSeat = NULL;
    WlSeat *pWlSeat_tmp = NULL;
    wl_list_for_each_safe(pWlSeat, pWlSeat_tmp, &pWLCtx->seats, link)
    {
        if (pWlSeat->seat_id == name)
        {
            mspin_lm_removeSeat(pWlSeat);
            break;
        }
    }
    pthread_mutex_unlock(&pWLCtx->lock_seat);

}
/*lint +e40*/
static struct wl_registry_listener registryListener = { mspin_lm_register, mspin_lm_remove };

static MSPIN_ERROR mspin_lm_createWLContext(mspin_layerManager_context_t *pLMContext)
{
    mspin_log_printLn(eMspinVerbosityDebug, "%s(ctx=%p) called", __FUNCTION__, pLMContext);

    if (!pLMContext)
    {
        mspin_log_printLn(eMspinVerbosityError, "%s(ctx=%p) ERROR: LayerManager context is NULL",
                __FUNCTION__, pLMContext);
        return MSPIN_ERROR_GENERAL;
    }

    if (!pLMContext->pWLCtx)
    {
        mspin_log_printLn(eMspinVerbosityError, "%s(ctx=%p) ERROR: Wayland context is NULL",
                __FUNCTION__, pLMContext);
        return MSPIN_ERROR_GENERAL;
    }

    mspin_wl_context_t *pWLCtx = pLMContext->pWLCtx;

    pWLCtx->p_wl_display = wl_display_connect(NULL);
    if (pWLCtx->p_wl_display == NULL)
    {
        mspin_log_printLn(eMspinVerbosityError, "%s(ctx=%p) ERROR: wl_display_connect failed", __FUNCTION__, pWLCtx);
        return MSPIN_ERROR_LAYERMANAGER;
    }

    wl_list_init(&pWLCtx->seats);
    pthread_mutex_init(&pWLCtx->lock_seat, NULL);

    pWLCtx->p_wl_input_queue = wl_display_create_queue(pWLCtx->p_wl_display);
    if (NULL == pWLCtx->p_wl_input_queue)
    {
        mspin_log_printLn(eMspinVerbosityError, "%s(ctx=%p) ERROR: Failed to create event queue",
                __FUNCTION__, pLMContext);
        return MSPIN_ERROR_LAYERMANAGER;
    }

    pWLCtx->p_wl_registry = wl_display_get_registry(pWLCtx->p_wl_display);
    wl_registry_add_listener(pWLCtx->p_wl_registry, &registryListener, pLMContext);
    wl_display_dispatch(pWLCtx->p_wl_display);

    int connectionID = wl_display_roundtrip(pWLCtx->p_wl_display);
    mspin_log_printLn(eMspinVerbosityDebug, "%s(ctx=%p) wl_display_roundtrip() returned first time %d",
            __FUNCTION__, pLMContext, connectionID); //not really required, only for debugging

    connectionID = wl_display_roundtrip(pWLCtx->p_wl_display); //Fix: Call a second time to fix timing issue reported 2015-10-22

    if (pWLCtx->p_wl_compositor == NULL)
    {
        mspin_log_printLn(eMspinVerbosityError, "%s(ctx=%p) ERROR: wl_compositor not set", __FUNCTION__, pLMContext);
        return MSPIN_ERROR_LAYERMANAGER;
    }

    if (0 == connectionID)
    {
        mspin_log_printLn(eMspinVerbosityError, "%s(ctx=%p) ERROR: connectionID is 0", __FUNCTION__, pLMContext);
        return MSPIN_ERROR_LAYERMANAGER;
    }

    pLMContext->pWLCtx->adapterSurfaceContext.shim_ctx = compositor_shim_initialize(pWLCtx->p_wl_display);
    if (!pLMContext->pWLCtx->adapterSurfaceContext.shim_ctx)
    {
        mspin_log_printLn(eMspinVerbosityError, "ERROR: compositor_shim_initialize failed", __FUNCTION__);
        return MSPIN_ERROR_LAYERMANAGER;
    }

    mspin_log_printLn(eMspinVerbosityVerbose, "%s(ctx=%p) done", __FUNCTION__, pLMContext);

    return MSPIN_SUCCESS;
}

/* PRQA: Lint Message 40: "link" used intentionally*/
/*lint -e40*/
static void mspin_lm_destroyWLContext(mspin_wl_context_t *pWLCtx)
{
    mspin_log_printLn(eMspinVerbosityDebug, "%s(ctx=%p)", __FUNCTION__, pWLCtx);

    pWLCtx->initialized = FALSE;

    pthread_mutex_lock(&pWLCtx->lock_seat);
    WlSeat *pWlSeat = NULL;
    WlSeat *pWlSeat_tmp = NULL;
    wl_list_for_each_safe(pWlSeat,pWlSeat_tmp, &pWLCtx->seats, link)
    {
        mspin_lm_removeSeat(pWlSeat);
    }
    pthread_mutex_unlock(&pWLCtx->lock_seat);

    if (pWLCtx->p_wl_compositor)
    {
        wl_compositor_destroy(pWLCtx->p_wl_compositor);
        pWLCtx->p_wl_compositor = NULL;
    }

    if (pWLCtx->p_wl_registry)
    {
        wl_registry_destroy(pWLCtx->p_wl_registry);
        pWLCtx->p_wl_registry = NULL;
    }

    if (pWLCtx->p_wl_input_queue)
    {
        wl_event_queue_destroy(pWLCtx->p_wl_input_queue);
        pWLCtx->p_wl_input_queue = NULL;
    }

    if(pWLCtx->p_wl_display)
    {
        wl_display_disconnect(pWLCtx->p_wl_display);
        pWLCtx->p_wl_display = NULL;
    }
    pthread_mutex_destroy(&pWLCtx->lock_seat);

    if (pWLCtx->pFramebufferContext)
    {
        free(pWLCtx->pFramebufferContext);
        pWLCtx->pFramebufferContext = 0;
    }
}
/*lint +e40*/

static void mspin_lm_destroySurface(mspin_layerManager_context_t *pLMContext)
{
    if (!pLMContext)
    {
        mspin_log_printLn(eMspinVerbosityError,
                "%s(pLMContext=%p) ERROR: pLMContext is NULL", __FUNCTION__, pLMContext);
        return;
    }

    if (compositor_shim_surface_destroy(&pLMContext->pWLCtx->adapterSurfaceContext)<0)
    {
        mspin_log_printLn(eMspinVerbosityDebug,
                       "%s(pLMContext=%p) ERROR: compositor_shim_surface_destroy failed! ", __FUNCTION__, pLMContext);
    }

    mspin_wl_context_t *pWLCtx = pLMContext->pWLCtx;

    if (pWLCtx->p_wl_surface)
    {
        mspin_log_printLn(eMspinVerbosityDebug,
                "%s(ctx=%p) delete surface=%p",
                __FUNCTION__, pLMContext, pWLCtx->p_wl_surface);

        wl_surface_destroy(pWLCtx->p_wl_surface);
        pWLCtx->p_wl_surface = NULL;
    }
}

static int mspin_lm_createImageBuffer(mspin_layerManager_context_t *pLMContext)
{

    if (!pLMContext)
    {
        mspin_log_printLn(eMspinVerbosityError, "%s(pLMContext=%p) ERROR: LM context is NULL",
                __FUNCTION__, pLMContext);
        return MSPIN_LM_FAIL;
    }

    if (!pLMContext->pWLCtx)
    {
        mspin_log_printLn(eMspinVerbosityError, "%s(pLMContext=%p) ERROR: pWLCtx is NULL",
                __FUNCTION__, pLMContext);
        return MSPIN_LM_FAIL;
    }

    mspin_wl_context_t *pWLCtx = pLMContext->pWLCtx;

    mspin_log_printLn(eMspinVerbosityDebug, "%s(pLMContext=%p)", __FUNCTION__, pLMContext);

    pWLCtx->p_wl_surface = wl_compositor_create_surface(pWLCtx->p_wl_compositor);
    if (!pWLCtx->p_wl_surface)
    {
        mspin_log_printLn(eMspinVerbosityError, "%s(pLMContext=%p) ERROR: Failed to create wl_surface",
                __FUNCTION__, pLMContext);
        return MSPIN_LM_FAIL;
    }

    mspin_log_printLn(eMspinVerbosityDebug, "%s(pLMContext=%p) wl_surface=%p created",
            __FUNCTION__, pLMContext,  pWLCtx->p_wl_surface);

    int c_shim_result = compositor_shim_surface_init(&pLMContext->pWLCtx->adapterSurfaceContext, pWLCtx->p_wl_surface, pLMContext->layerId, pLMContext->surfaceId,
            (uint32_t) pLMContext->frameWidth,(uint32_t) pLMContext->frameHeight);

    if(c_shim_result < 0)
    {
        mspin_log_printLn(eMspinVerbosityError, "%s(pLMContext=%p) ERROR: compositor_shim_surface_init failed",
                __FUNCTION__, pLMContext);
        return MSPIN_LM_FAIL;
    }

    if(pWLCtx->force_fullscreen)
    {
        pLMContext->pWLCtx->adapterSurfaceContext.destWidth = pWLCtx->screen_width;
        pLMContext->pWLCtx->adapterSurfaceContext.destHeight = pWLCtx->screen_height;
    }
    else
    {
        float screen_aspect = (float)pWLCtx->screen_width / pWLCtx->screen_height;
        float phone_aspect = (float)pLMContext->frameWidth / pLMContext->frameHeight;

        if(screen_aspect < phone_aspect)
        {
            pLMContext->pWLCtx->adapterSurfaceContext.destWidth = pWLCtx->screen_width;
            pLMContext->pWLCtx->adapterSurfaceContext.destHeight = (uint32_t)((float)pWLCtx->screen_width / phone_aspect);
        }
        else
        {
            pLMContext->pWLCtx->adapterSurfaceContext.destHeight = pWLCtx->screen_height;
            pLMContext->pWLCtx->adapterSurfaceContext.destWidth = (uint32_t)((float)pWLCtx->screen_height * phone_aspect);
        }
    }

    pLMContext->pWLCtx->adapterSurfaceContext.destX = (pWLCtx->screen_width-pLMContext->pWLCtx->adapterSurfaceContext.destWidth)/2 + pWLCtx->screen_x_offset;
    pLMContext->pWLCtx->adapterSurfaceContext.destY = (pWLCtx->screen_height-pLMContext->pWLCtx->adapterSurfaceContext.destHeight)/2 + pWLCtx->screen_y_offset;

    c_shim_result = compositor_shim_surface_configure(pLMContext->pWLCtx->adapterSurfaceContext.shim_ctx,&pLMContext->pWLCtx->adapterSurfaceContext,ADAPTER_CONFIGURATION_ALL);

    if (c_shim_result<0)
    {
        mspin_log_printLn(eMspinVerbosityError, "%s(pLMContext=%p) ERROR: compositor_shim_surface_configure() failed",
                __FUNCTION__, pLMContext);
        return MSPIN_LM_FAIL;
    }

    if (!mspin_framebuffer_createImage(pWLCtx, pLMContext->frameWidth, pLMContext->frameHeight))
    {
        mspin_log_printLn(eMspinVerbosityError, "%s(pLMContext=%p) ERROR: mspin_lm_createImage() failed",
                __FUNCTION__, pLMContext);
        return MSPIN_LM_FAIL;
    }

    return MSPIN_LM_SUCCESS;
}


static void mspin_lm_freeImageBuffer(mspin_layerManager_context_t *pLMContext)
{
    if (!pLMContext)
    {
        mspin_log_printLn(eMspinVerbosityError, "%s(pLMContext=%p) ERROR: LM context is NULL",
                __FUNCTION__, pLMContext);
        return;
    }

    if (!pLMContext->pWLCtx)
    {
        mspin_log_printLn(eMspinVerbosityError, "%s(pLMContext=%p) ERROR: Wayland context is NULL",
                __FUNCTION__, pLMContext);
        return;
    }

    pLMContext->pWLCtx->initialized = FALSE;
#ifdef MSPIN_X86SIM
    free(pLMContext->pWLCtx->p_image);
#endif

    mspin_lm_destroySurface(pLMContext);
}

static U32 mspin_lm_serverInit(mspin_layerManager_context_t *pLMContext, const char *desktopName)
{
    U32 ret = MSPIN_LM_FAIL;

    MSPIN_UNUSED(desktopName);

    mspin_log_printLn(eMspinVerbosityDebug, "%s(ctx=%p)", __FUNCTION__, pLMContext);

    if (pLMContext && pLMContext->pWLCtx)
    {
        if (MSPIN_SUCCESS != mspin_lm_createWLContext(pLMContext))
        {
            mspin_log_printLn(eMspinVerbosityError,
                    "%s(ctx=%p) ERROR: could not initialize LayerManager",
                    __FUNCTION__, pLMContext);
        }
        else
        {
            if (!mspin_lm_createImageBuffer(pLMContext))
            {
                mspin_log_printLn(eMspinVerbosityError,
                        "%s(ctx=%p) ERROR: could not create image buffer",
                        __FUNCTION__, pLMContext);
            }
            else
            {
                ret = MSPIN_LM_SUCCESS;
            }
        }
    }
    else
    {
        mspin_log_printLn(eMspinVerbosityError,
                "%s(ctx=%p) ERROR: Wayland context is NULL",
                __FUNCTION__, pLMContext);
    }

    return ret;
}

static int mspin_lm_setSurfaceFormat(mspin_wl_context_t *pWLCtx, MSPIN_PIXEL_FORMAT pixel_format)
{
    if (!pWLCtx)
    {
        mspin_log_printLn(eMspinVerbosityError, "%s(wlCtx=%p, pixel_format=%d) ERROR: wlCtx is NULL",
                __FUNCTION__, pWLCtx, (int)pixel_format);
        return MSPIN_LM_FAIL;
    }

    if (MSPIN_PIXEL_FORMAT_RGB565 == pixel_format)
    {
        pWLCtx->surface_format = APX_PIXELFORMAT_RGB_565;
    }
    else if (MSPIN_PIXEL_FORMAT_RGBA8888 == pixel_format)
    {
        mspin_log_printLn(eMspinVerbosityFatal, "%s(wlCtx=%p, pixel_format=%d) FATAL ERROR: pixel format not supported",
                __FUNCTION__, pWLCtx, (int)pixel_format);
        return MSPIN_LM_FAIL;
    }
    else if (MSPIN_PIXEL_FORMAT_ARGB8888 == pixel_format)
    {
        // Note: the enum ILM_PIXELFORMAT_RGBA_8888/APX_PIXELFORMAT_RGBA_8888 is in fact ARGB (with APX)!
        // At the moment ARGB is the only 32 bit format supported by APX
        pWLCtx->surface_format = APX_PIXELFORMAT_RGBA_8888;
    }
    else
    {
        mspin_log_printLn(eMspinVerbosityError, "%s(wlCtx=%p, pixel_format=%d) ERROR: pixel format unknown",
                __FUNCTION__, pWLCtx, (int)pixel_format);
        return MSPIN_LM_FAIL;
    }

    return MSPIN_LM_SUCCESS;
}

static mspin_wl_context_t* mspin_lm_initWLContext(MSPIN_PIXEL_FORMAT pixel_format,
        t_ilm_int screen_x_offset, t_ilm_int screen_y_offset,
        t_ilm_int screen_width, t_ilm_int screen_height,
        t_ilm_bool force_fullscreen, enum mspin_lm_flags flags)
{
    mspin_wl_context_t *pWLCtx = malloc(sizeof(mspin_wl_context_t));
    if (pWLCtx)
    {
        memset(pWLCtx, 0, sizeof(mspin_wl_context_t));

        if (!mspin_framebuffer_init(pWLCtx))
        {
            mspin_log_printLn(eMspinVerbosityError, "%s() ERROR: mspin_lm_apx_init failed",
                    __FUNCTION__);

            free(pWLCtx);
            pWLCtx = NULL;
            return NULL;
        }

        pWLCtx->screen_x_offset = screen_x_offset;
        pWLCtx->screen_y_offset = screen_y_offset;
        pWLCtx->screen_width = screen_width;
        pWLCtx->screen_height = screen_height;
        pWLCtx->force_fullscreen = force_fullscreen;
        pWLCtx->flags = flags;

        U32 id = 0;
        for (id = 0; id < MSPIN_WL_MAX_TOUCHPOINTS; ++id)
        {
            pWLCtx->touchInfo[id].valid = FALSE;
            pWLCtx->touchInfo[id].lastTimeAt = 0;
            pWLCtx->touchEv[id].event = eTOUCHTYPE_Cancelled;
            pWLCtx->touchEv[id].xPosition = 0;
            pWLCtx->touchEv[id].yPosition = 0;
            pWLCtx->touchEv[id].fingerId = id;
            pWLCtx->lasttimeup1 = 0;
        }
        pWLCtx->lastPointerX = 0;
        pWLCtx->lastPointerY = 0;
        pWLCtx->lastSentPointerX = 0;
        pWLCtx->lastSentPointerY = 0;
        pWLCtx->pointerButtonState = 0;
        pWLCtx->pointerButtonDownToBeSent = FALSE;
        pWLCtx->lastTimePointerEventSent = 0;

        if (!mspin_lm_setSurfaceFormat(pWLCtx, pixel_format))
        {
            mspin_log_printLn(eMspinVerbosityError, "%s() ERROR: mspin_lm_setSurfaceFormat() failed",
                    __FUNCTION__);
            free(pWLCtx);
            pWLCtx = NULL;
            return NULL;
        }

        mspin_log_printLn(eMspinVerbosityDebug, "%s() Wayland context=%p created", __FUNCTION__, pWLCtx);
    }
    return pWLCtx;
}

MSPIN_ERROR mspin_lm_init(void)
{
    MSPIN_ERROR ret = MSPIN_SUCCESS;

    if (compositor_shim_wait_server() < 0)
    {
        mspin_log_printLn(eMspinVerbosityError, "%s() ERROR: compositor_shim_wait_server() failed", __FUNCTION__);
        ret = MSPIN_ERROR_LAYERMANAGER;
    }else
    {
        mspin_log_printLn(eMspinVerbosityDebug, "%s() compositor_shim_wait_server() succeeded", __FUNCTION__);
    }

#ifdef MYSPIN_LM_GIT_VERSION
    mspin_log_printLn(eMspinVerbosityInfo, "%s() APX LM Adapter version %s", __FUNCTION__, MYSPIN_LM_GIT_VERSION);
#else
    mspin_log_printLn(eMspinVerbosityInfo, "%s() APX LM Adapter version %s", __FUNCTION__, "n/a");
#endif

#ifdef MSPIN_TOUCH_DELAY
    mspin_lm_startDelayWorker();
#endif

    return ret;
}


void mspin_lm_shutdown(void)
{

    if (compositor_shim_close_server() < 0)
    {
        mspin_log_printLn(eMspinVerbosityError, "%s() ERROR: compositor_shim_close_server() failed", __FUNCTION__);
    }
    else
    {
        mspin_log_printLn(eMspinVerbosityDebug, "%s() compositor_shim_close_server() succeeded", __FUNCTION__);
    }
}


static void mspin_lm_letPollExit(mspin_layerManager_context_t *pLMContext)
{
    if (!pLMContext || !pLMContext->pWLCtx)
    {
        mspin_log_printLn(eMspinVerbosityError,
                "%s(ctx=%p) ERROR: LM/WL context is NULL",
                __FUNCTION__, pLMContext);
        return;
    }

    int loopCnt = 0;
    for (loopCnt = 0; loopCnt < 5; ++loopCnt)
    {
       int rc = pthread_rwlock_unlock(&(pLMContext->rwlock));
       if (rc)
       {
           mspin_log_printLn(eMspinVerbosityFatal, "%s(pLMCtx=%p) FATAL ERROR: rwlock unlock failed with rc=%d",
                   __FUNCTION__, pLMContext, rc);
       }

       mspin_log_printLn(eMspinVerbosityInfo, "%s(ctx=%p) sleeping 60ms ... (%d/5)", __FUNCTION__, pLMContext, loopCnt+1);
       usleep(60*1000);

       rc = pthread_rwlock_wrlock(&(pLMContext->rwlock));
       if (rc)
       {
           mspin_log_printLn(eMspinVerbosityFatal, "%s(pLMCtx=%p) FATAL ERROR: rwlock failed with rc=%d",
                   __FUNCTION__, pLMContext, rc);
       }

       if (pLMContext->pWLCtx->pollingActive == false)
       {
          break;
       }
    }
}


// when called, rwlock must already be locked
static void mspin_lm_deleteContextIntern(mspin_layerManager_context_t *pLMContext)
{
    if (!pLMContext)
    {
        mspin_log_printLn(eMspinVerbosityError,
                "%s(ctx=%p) ERROR: LM context is NULL",
                __FUNCTION__, pLMContext);
        return;
    }

    // rwlock is already locked
    if (pLMContext->pWLCtx)
    {
      pLMContext->readyToUse = FALSE;

      if (pLMContext->pWLCtx->pollingActive)
      {
         mspin_lm_letPollExit(pLMContext);
      }

      if (pLMContext->pWLCtx->pollingActive)
      {
         mspin_log_printLn(eMspinVerbosityWarn,
               "%s(ctx=%p) done from thread=%lu with error: poll active - cannot clean up - leak",
               __FUNCTION__, pLMContext, (long unsigned int) pthread_self());
      }
      else
      {
         mspin_framebuffer_destroyImage(pLMContext->pWLCtx);

         mspin_lm_freeImageBuffer(pLMContext);
         if(compositor_shim_terminate(pLMContext->pWLCtx->adapterSurfaceContext.shim_ctx)<0)
         {
             mspin_log_printLn(eMspinVerbosityError,
               "%s(shim_ctx=%p) ERROR: compositor_shim_terminate() failed",
                __FUNCTION__, pLMContext->pWLCtx->adapterSurfaceContext.shim_ctx);
         }
         mspin_lm_destroyWLContext(pLMContext->pWLCtx);

         free(pLMContext->pWLCtx);
         pLMContext->pWLCtx = NULL;

         mspin_log_printLn(eMspinVerbosityInfo,
               "%s(ctx=%p) done from thread=%lu", __FUNCTION__, pLMContext,
               (long unsigned int) pthread_self());
      }
    }
    else
    {
        mspin_log_printLn(eMspinVerbosityError,
                "%s(ctx=%p) ERROR: Wayland context is NULL",
                __FUNCTION__, pLMContext);
    }
}


void mspin_lm_deleteContext(mspin_layerManager_context_t *pLMContext)
{
    if (!pLMContext)
    {
        mspin_log_printLn(eMspinVerbosityError,
                "%s(ctx=%p) ERROR: LM context is NULL",
                __FUNCTION__, pLMContext);
        return;
    }
    else
    {
        mspin_log_printLn(eMspinVerbosityDebug,
                "%s(ctx=%p) entered",
                __FUNCTION__, pLMContext);
    }

    int rc = pthread_rwlock_wrlock(&(pLMContext->rwlock));
    if (rc)
    {
        mspin_log_printLn(eMspinVerbosityFatal, "%s(pLMCtx=%p) FATAL ERROR: rwlock failed with rc=%d",
                __FUNCTION__, pLMContext, rc);
    }

    mspin_lm_deleteContextIntern(pLMContext);

    rc = pthread_rwlock_unlock(&(pLMContext->rwlock));
    if (rc)
    {
        mspin_log_printLn(eMspinVerbosityFatal, "%s(pLMCtx=%p) FATAL ERROR: rwlock unlock failed with rc=%d",
                __FUNCTION__, pLMContext, rc);
    }

}


MSPIN_ERROR mspin_lm_createContext(mspin_layerManager_context_t *pLMContext, void *pCoreInstance)
{
    MSPIN_ERROR result = MSPIN_SUCCESS;

    if (!pLMContext)
    {
        mspin_log_printLn(eMspinVerbosityError, "%s(ctx=%p) ERROR: pLMContext is NULL",
                __FUNCTION__, pLMContext);
        return MSPIN_ERROR_GENERAL;
    }

    //Error out in case of LayerManager is already present
    if (pLMContext->pWLCtx)
    {
        mspin_log_printLn(eMspinVerbosityFatal,
                "%s(ctx=%p) FATAL ERROR: Old LayerManager present (called from thread=%lu)",
                __FUNCTION__, pLMContext, (unsigned long int)pthread_self());
        return MSPIN_ERROR_GENERAL;
    }

    mspin_log_printLn(eMspinVerbosityDebug, "%s(context=%p) -> wait for lock.",
            __FUNCTION__, pLMContext);
    int rc = pthread_rwlock_wrlock(&(pLMContext->rwlock));
    if (rc)
    {
        mspin_log_printLn(eMspinVerbosityFatal, "%s(pLMCtx=%p) FATAL ERROR: rwlock failed with rc=%d",
                __FUNCTION__, pLMContext, rc);
    }
    else
    {
       mspin_log_printLn(eMspinVerbosityDebug, "%s(context=%p) lock retrieved",
             __FUNCTION__, pLMContext);
    }

    //Start LayerManager adapter
    enum mspin_lm_flags flags = MSPIN_WL_ENABLE_TOUCHEVENTS;
    if (pLMContext->useWaylandTouch == FALSE)
    {
        flags = MSPIN_WL_DISABLE_TOUCHEVENTS;
    }

    pLMContext->pCoreHandle = pCoreInstance;

    pLMContext->pWLCtx = mspin_lm_initWLContext(pLMContext->pixelFormat, 0, 0,
            pLMContext->frameWidth, pLMContext->frameHeight, 0, flags);

    if (pLMContext->pWLCtx == NULL)
    {
        mspin_log_printLn(eMspinVerbosityError,
                "%s(context=%p) ERROR: Cannot initialize WL!",
                __FUNCTION__, pLMContext);

        rc = pthread_rwlock_unlock(&(pLMContext->rwlock));
        if (rc)
        {
            mspin_log_printLn(eMspinVerbosityFatal, "%s(pLMCtx=%p) FATAL ERROR: rwlock unlock failed with rc=%d",
                    __FUNCTION__, pLMContext, rc);
        }
        return MSPIN_ERROR_LAYERMANAGER;
    }

    if (!mspin_lm_serverInit(pLMContext, ""))
    {
        mspin_log_printLn(eMspinVerbosityError,
                "%s(context=%p) ERROR: Failed to init WL server", __FUNCTION__,
                pLMContext);
        mspin_lm_deleteContextIntern(pLMContext);
        result = MSPIN_ERROR_LAYERMANAGER;
    }
    else
    {
        mspin_log_printLn(eMspinVerbosityDebug,
                "%s(context=%p) Wayland initialized", __FUNCTION__, pLMContext);
    }

    pLMContext->readyToUse = TRUE;

    mspin_log_printLn(eMspinVerbosityInfo,
            "%s(context=%p) LayerManager ready to use (called from thread=%lu)",
            __FUNCTION__, pLMContext, (long unsigned int)pthread_self());

    rc = pthread_rwlock_unlock(&(pLMContext->rwlock));
    if (rc)
    {
        mspin_log_printLn(eMspinVerbosityFatal, "%s(pLMCtx=%p) FATAL ERROR: rwlock unlock failed with rc=%d",
                __FUNCTION__, pLMContext, rc);
    }

    return result;
}


void mspin_lm_pumpEvents(mspin_layerManager_context_t *pLMContext, S32 timeout)
{
    if (!pLMContext)
    {
        mspin_log_printLn(eMspinVerbosityError,
                "%s(ctx=%p, timeout=%d) ERROR: MSPIN context is NULL",
                __FUNCTION__, pLMContext, timeout);
        return;
    }

    int rc = pthread_rwlock_rdlock(&(pLMContext->rwlock)); // rdlock - see comment on top
    if (rc)
    {
        mspin_log_printLn(eMspinVerbosityFatal, "%s(pLMCtx=%p) FATAL ERROR: rwlock failed with rc=%d",
                __FUNCTION__, pLMContext, rc);
    }
    if (pLMContext->pWLCtx && pLMContext->readyToUse)
    {
        int result = 0;
        struct pollfd pfd[1];

        mspin_log_printLn(eMspinVerbosityVerbose,
                "%s(ctx=%p, timeout=%dms)",
                __FUNCTION__, pLMContext, timeout);

        pfd[0].fd = wl_display_get_fd(pLMContext->pWLCtx->p_wl_display);
        pfd[0].events = POLLIN;
        
#ifndef MSPIN_X86SIM
        //Get exclusive access to display file descriptor
        while (0 != wl_display_prepare_read_queue(pLMContext->pWLCtx->p_wl_display,
                pLMContext->pWLCtx->p_wl_input_queue))
        {
            mspin_log_printLn(eMspinVerbosityWarn,
                    "%s(ctx=%p, timeout=%d) WARNING: wl_display_prepare_read_queue() failed",
                    __FUNCTION__, pLMContext, timeout);
            wl_display_dispatch_queue_pending(pLMContext->pWLCtx->p_wl_display,
                    pLMContext->pWLCtx->p_wl_input_queue);
        }
#endif //#ifndef MSPIN_X86SIM

        //Flush events before waiting for new events
        wl_display_flush(pLMContext->pWLCtx->p_wl_display);

        pLMContext->pWLCtx->pollingActive = true;

        rc = pthread_rwlock_unlock(&(pLMContext->rwlock)); // don't want to block the lock all the time
        if (rc)
        {
            mspin_log_printLn(eMspinVerbosityFatal, "%s(pLMCtx=%p) FATAL ERROR: rwlock unlock failed with rc=%d",
                    __FUNCTION__, pLMContext, rc);
        }
        //Wait for Wayland event (not necessarily input event, blocking)
        result = poll(pfd, 1, timeout);

        rc = pthread_rwlock_rdlock(&(pLMContext->rwlock));
        pLMContext->pWLCtx->pollingActive = false;
        if (rc)
        {
            mspin_log_printLn(eMspinVerbosityFatal, "%s(pLMCtx=%p) FATAL ERROR: rwlock failed with rc=%d",
                    __FUNCTION__, pLMContext, rc);
        }
        if (!pLMContext->pWLCtx || !pLMContext->readyToUse)
        {
            mspin_log_printLn(eMspinVerbosityDebug,
                    "%s(ctx=%p, timeout=%dms) During polling the LayerManager got deinitialized",
                    __FUNCTION__, pLMContext, timeout);

            wl_display_cancel_read(pLMContext->pWLCtx->p_wl_display);
        }
        else
        {
#ifndef MSPIN_X86SIM
            if (result < 1)
            {
                //When something goes wrong or timeout (=no events) => release exclusive access to display file descriptor
                wl_display_cancel_read(pLMContext->pWLCtx->p_wl_display);
            }
            else
            {
                //Read and dispatch events
                wl_display_read_events(pLMContext->pWLCtx->p_wl_display);
                wl_display_dispatch_queue_pending(pLMContext->pWLCtx->p_wl_display,
                        pLMContext->pWLCtx->p_wl_input_queue);
            }
#else
            MSPIN_UNUSED(result);
            if (pfd[0].revents & POLLIN)
            {
                wl_display_dispatch(pLMContext->pWLCtx->p_wl_display);
            }
            else
            {
                wl_display_dispatch_pending(pLMContext->pWLCtx->p_wl_display);
            }
#endif //#ifndef MSPIN_X86SIM
        }
        rc = pthread_rwlock_unlock(&(pLMContext->rwlock));
        if (rc)
        {
            mspin_log_printLn(eMspinVerbosityFatal, "%s(pLMCtx=%p) FATAL ERROR: rwlock unlock failed with rc=%d",
                    __FUNCTION__, pLMContext, rc);
        }
    }
    else
    {
       rc = pthread_rwlock_unlock(&(pLMContext->rwlock));
       if (rc)
       {
           mspin_log_printLn(eMspinVerbosityFatal, "%s(pLMCtx=%p) FATAL ERROR: rwlock unlock failed with rc=%d",
                   __FUNCTION__, pLMContext, rc);
       }
       if (!pLMContext->pWLCtx)
       {
           //This is ok, when Wayland is not yet created
           mspin_log_printLn(eMspinVerbosityVerbose,
                   "%s(ctx=%p, timeout=%d) Wayland context is NULL",
                   __FUNCTION__, pLMContext, timeout);
           usleep(timeout*1000); //wait timeout ms
       }
       else
       {
           //This is ok, when Wayland is not yet initialized
           mspin_log_printLn(eMspinVerbosityVerbose,
                   "%s(ctx=%p, timeout=%d) Wayland not ready!",
                   __FUNCTION__, pLMContext, timeout);
           usleep(timeout*1000); //wait timeout ms
       }
    }
}

static U8* mspin_lm_lockRectangle(mspin_layerManager_context_t *pLMContext,
        U16 x, U16 y, size_t *pStride)   //x,y of topleft corner
{

    if (!pLMContext)
    {
        mspin_log_printLn(eMspinVerbosityFatal,
                "%s(pLMCtx=%p) FATAL ERROR: LayerManager context is NULL",
                __FUNCTION__, pLMContext);
        return NULL;
    }

    U8 * ptr = NULL;

    if (pLMContext->pWLCtx)
    {
        if (pLMContext->pWLCtx->initialized == 1)
        {
            *pStride = pLMContext->pWLCtx->stride;

            ptr = pLMContext->pWLCtx->p_image[0] + x * pLMContext->bytesPerPixel
                    + y * pLMContext->pWLCtx->stride;
        }
    }

    return ptr;
}

void mspin_lm_copyRect(mspin_layerManager_context_t *pLMContext, U8 currentNumber, U16 x0, U16 y0,
        U16 width, U16 height, U8 *pBuffer, U32 bufferSize)
{
    if (!pLMContext)
    {
        mspin_log_printLn(eMspinVerbosityFatal,
                "%s(pLMCtx=%p) FATAL ERROR: LayerManager context is NULL",
                __FUNCTION__, pLMContext);
        return;
    }

    mspin_log_printLn(eMspinVerbosityVerbose, "%s(ctx=%p, num=%d, size=%d) wait for lock",
               __FUNCTION__, pLMContext, currentNumber, bufferSize);
    int rc = pthread_rwlock_wrlock(&(pLMContext->rwlock));
    if (rc)
    {
        mspin_log_printLn(eMspinVerbosityFatal, "%s(pLMCtx=%p) FATAL ERROR: rwlock failed with rc=%d",
                __FUNCTION__, pLMContext, rc);
    }
    else
    {
       mspin_log_printLn(eMspinVerbosityVerbose, "%s(ctx=%p, num=%d, size=%d) got lock -> continue",
                   __FUNCTION__, pLMContext, currentNumber, bufferSize);
    }

    if (pLMContext->pWLCtx)
    {
        U16 i = 0;
        size_t stride = 0;
        U8* pImage = NULL;

        pImage = mspin_lm_lockRectangle(pLMContext, x0, y0, &stride);
        if (pImage)
        {
            // copy image
            for (i = 0; i < height; ++i)
            {
                memcpy(pImage, pBuffer,
                        width * pLMContext->bytesPerPixel);
                pImage += stride;
                pBuffer += width * pLMContext->bytesPerPixel;
            }
        }
        else
        {
            mspin_log_printLn(eMspinVerbosityFatal, "%s(ctx=%p, num=%d, size=%d) ERROR: pImage is NULL",
                    __FUNCTION__, pLMContext, currentNumber, bufferSize);
        }
    }
    else
    {
        mspin_log_printLn(eMspinVerbosityFatal, "%s(ctx=%p, num=%d, size=%d) FATAL ERROR: Wayland context is NULL",
                __FUNCTION__, pLMContext, currentNumber, bufferSize);
    }
    rc = pthread_rwlock_unlock(&(pLMContext->rwlock));
    if (rc)
    {
        mspin_log_printLn(eMspinVerbosityFatal, "%s(pLMCtx=%p) FATAL ERROR: rwlock unlock failed with rc=%d",
                __FUNCTION__, pLMContext, rc);
    }
}


void mspin_lm_drawFrame(mspin_layerManager_context_t *pLMContext)
{
    if (!pLMContext)
    {
        mspin_log_printLn(eMspinVerbosityFatal,
               "%s(pLMCtx=%p) FATAL ERROR: LayerManager context is NULL",
               __FUNCTION__, pLMContext);
        return;
    }

    mspin_log_printLn(eMspinVerbosityVerbose, "%s(pLMCtx=%p)",
            __FUNCTION__, pLMContext);

    int rc = pthread_rwlock_rdlock(&(pLMContext->rwlock)); // rdlock - see comment on top
    if (rc)
    {
        mspin_log_printLn(eMspinVerbosityFatal, "%s(pLMCtx=%p) FATAL ERROR: rwlock failed with rc=%d",
                __FUNCTION__, pLMContext, rc);
    }
    if (pLMContext->pWLCtx)
    {
        //mspin_lm_frameBufferUpdateEnd(pMspinContext->pWLCtx);
        mspin_framebuffer_drawImage(pLMContext->pWLCtx, pLMContext->frameWidth, pLMContext->frameHeight);

#ifdef MSPIN_X86SIM
        //        glClearColor(1.0f, 0.0f, 0.0f, 1.0f);
        //        glClear(GL_COLOR_BUFFER_BIT);
        //        mspin_wl_if_swap_buffer(pContext->pWLCtx);
        ////        eglSwapBuffers(pLMContext->pWLCtx->egl_display, p_wl_ctx->egl_surface);
        //        sleep(2);
#endif
    }
    else
    {
        mspin_log_printLn(eMspinVerbosityFatal, "%s(pLMCtx=%p) FATAL ERROR: pWLCtx context is NULL",
                __FUNCTION__, pLMContext);
    }
    rc = pthread_rwlock_unlock(&(pLMContext->rwlock));
    if (rc)
    {
        mspin_log_printLn(eMspinVerbosityFatal, "%s(pLMCtx=%p) FATAL ERROR: rwlock unlock failed with rc=%d",
                __FUNCTION__, pLMContext, rc);
    }
}

/*lint +e160 +e505 +e578 +e413 +e613*/
